clear; clc
%% Path 
addpath('Config\')
addpath('Vehicle data\')
addpath('R0 and COP\')

%% User input
% Time Step
Ts = 0.2;
% Environment temperature
T_init = 26;
% Driving cycle
cycle = 'UDDS';

%Target battery temperature
T_target = 30; %°C

%% Parameters to run simulation
% Load vehicle paramaters from MATWORKS example
run ElectricVehicleThermalManagementParameters.m
% Load other vehicle parameters (COP, battery thermal parameters, ecc..)
run R0_COP.m
% Initial conditions
run Initial_conditions.m
% MPC parameters
run MPC_parameters.m
%% Use the backward simulator current
if strcmp(cycle,'BuildIn')
    use_simulator_current = 0;
    t = 1;
    C_rate = 1;
    t_end = 3600;
    v_vehicle = 0;
else
    use_simulator_current = 1;
    load(['../gambero/Results/BattCurrProfile/',cycle],'prof','veh');
    t = [prof.time]';
    C_rate = [prof.battCurr]'/veh.batt.nomCap;
    t_end = t(end);
    v_vehicle = [prof.vehSpd]' * 3.6; % km/h
end

% Strategy (AMPC or Reactive)
strategy = 'AMPC';
model_name = append('EVThermalManagement_',strategy,'_2020b_new_comp');
open(model_name)


% Current
I_traction = C_rate * battery_Ah_capacity;


if strcmp(strategy,'AMPC')
    %% Write symbolic jacobian matrices, which means write the initial linearized model
    % Define states, outputs and inputs
    syms x [1 1] % States (x1 = battery temperature)
    syms y [1 1] % Output (y1 = battery temperature)
    syms u [3 1] % Input (u1 = compressor power, u2 = current, u3 = environment temperature)  
    
    %syms thetaB SOC iB Pbtm
    syms R0 c % Define internal resistance R0 and coefficient of performance COP
    R0 = poly2sym(r_pol_coef,x1); % Cell R0 = f(battery temperature); 
    c = c1*u3 + c0; % c is the COP: c = Qdot_BTM/Pcomp. c = f(environment temperature), 
    
    % States derivative
    syms xdot1 % States derivatives
    xdot1 = (Joule_incr_factor*battery_N_cells_per_module*battery_N_modules*R0*(u2+u1/battery_V_nom)^2 ...
    - c*u1) / (battery_mass*battery_cell_cp);
    xdot = [xdot1];
    
    % Output
    y1 = x1;
    
    % Jacobians
    As = [diff(xdot1,x1)];
    Bs = [diff(xdot1,u1), diff(xdot1,u2), diff(xdot1,u3)];
    Cs = [diff(y1,x1)];
    Ds = [diff(y1,u1), diff(y1,u2), diff(y1,u3)];
    
    %% Symbolic function generation 
    % Creation of Jacobian, R0, and COP functions in Simulink ( block "Jacobians" (Compressor control subsystem), "R0" (interal model subsystem), "c" (interal model subsystem))
    matlabFunctionBlock(append(model_name,'/Controls/Compressor Control/Jacobians'),As,Bs,xdot)
    matlabFunctionBlock(append(model_name,'/Internal Model AMPC/R0'),R0)
    matlabFunctionBlock(append(model_name,'/Internal Model AMPC/c'),c)
    
    
    %% Creation of mpcobj
    % Write the initial condition x0 u0 y0
    x0 = [T_battery_init]; % look up file "ElectricVehicleManagementParamenters" initial condition block
    u0 = [EPowerCompressor_init; i_battery_init; T_env_init];
    y0 = [T_battery_init];
    
    % Define the nominal initial point 
    x1 = sym(x0(1));
    u1 = sym(u0(1));
    u2 = sym(u0(2));
    u3 = sym(u0(3));
    % Matrixes to create the plant model
    A = double(subs(As));
    B = double(subs(Bs));
    C = double(subs(Cs));
    D = double(subs(Ds));
    
    % Continuous and discrete time plant models
    plantCT = ss(A,B,C,D);
    plantDT = c2d(plantCT,Ts_MPC);
    % Set disturbances, manipulated variables, output
    plantDT.InputGroup.MeasuredDisturbances = [2 3];
    plantDT.InputGroup.ManipulatedVariables = 1;
    plantDT.OutputGroup.Measured = 1;
    plantDT.InputName = {'Pbtm','iB','Tenv'};
    plantDT.OutputName = {'Tbattery'};
    
    % Create MPC object
    mpcobj = mpc(plantDT,Ts_MPC,p,pc);
    
    % Nominal starting point
    mpcobj.Model.Nominal = struct('X', x0, 'U', u0, 'Y', y0, 'DX', [0]);
    
    % Output variable
    % Batt temperature
    mpcobj.OV(1).Min = min_batt_T;
    mpcobj.OV(1).MinECR = 0; % 0 is hard constraint, the higher the softer the constraint
    mpcobj.OV(1).Max = max_batt_T;
    mpcobj.OV(1).MaxECR = 0; % 0 is hard constraint, the higher the softer the constraint
    mpcobj.OV(1).ScaleFactor = T_batt_scale;
    
    % Control variable 
    % Compressor power
    mpcobj.MV.Max = MaxEPowerCompressor;
    mpcobj.MV.Min = 0;
    mpcobj.MV.ScaleFactor = MaxEPowerCompressor;
    mpcobj.MV.Target = 0; % We want that the compressor power is kept minimum
    mpcobj.MV.RateMin = Pcomp_RateMin;
    mpcobj.MV.RateMax = Pcomp_RateMax;
    
    % Disturbance variable
    mpcobj.DV(1).ScaleFactor = current_scale; % current 
    mpcobj.DV(2).ScaleFactor = T_env_scale; % environment temperature
    
    % Cost function weights
    mpcobj.Weights.OutputVariables = [WeightTbattery*ones(p,1)];
    mpcobj.Weights.ManipulatedVariables = WeightEPowerComp*ones(p,1);
    mpcobj.Weights.ManipulatedVariablesRate = WeightEPowerRateComp*ones(p,1);
    mpcobj.Weights.ECR = WeightECR;
end


%% Run the simulation
tic 
sim('EVThermalManagement_AMPC_2020b_new_comp') %('Prova')%
toc

%% Plot results
addpath('../Plot/')
run('../Plot/postprocessing.m')
plot_prof(time,battery,EPowers,ElectricEnergy)

%% Save results
save_results(strategy,T_init,cycle,time,Distance,ElectricEnergy,EPowers, battery) %,chiller,chillerByPassValve, compressor, condenser,simlog

